; ====================================================================================================
;
; common.asm
;
; included from main.asm
;
; ====================================================================================================

SECTION code_user

; ■キースキャン用定数
KEY_ON:                 EQU $01     ; キーオン
KEY_OFF:                EQU $00     ; キーオフ

KEY_A:                  EQU $26     ; Aキー
KEY_B:                  EQU $27     ; Bキー
KEY_C:                  EQU $30     ; Cキー
KEY_D:                  EQU $31     ; Dキー
KEY_E:                  EQU $32     ; Eキー
KEY_F:                  EQU $33     ; Fキー
KEY_G:                  EQU $34     ; Gキー
KEY_H:                  EQU $35     ; Hキー
KEY_I:                  EQU $36     ; Iキー
KEY_J:                  EQU $37     ; Jキー
KEY_K:                  EQU $40     ; Kキー
KEY_L:                  EQU $41     ; Lキー
KEY_M:                  EQU $42     ; Mキー
KEY_N:                  EQU $43     ; Nキー
KEY_O:                  EQU $44     ; Oキー
KEY_P:                  EQU $45     ; Pキー
KEY_Q:                  EQU $46     ; Qキー
KEY_R:                  EQU $47     ; Rキー
KEY_S:                  EQU $50     ; Sキー
KEY_T:                  EQU $51     ; Tキー
KEY_U:                  EQU $52     ; Uキー
KEY_V:                  EQU $53     ; Vキー
KEY_W:                  EQU $54     ; Wキー
KEY_X:                  EQU $55     ; Xキー
KEY_Y:                  EQU $56     ; Yキー
KEY_Z:                  EQU $57     ; Zキー
KEY_F1:                 EQU $65     ; F1キー
KEY_F5:                 EQU $71     ; F5キー

; ====================================================================================================
; キー入力値取得サブルーチン
; @ToDo : 直前の入力値を退避するようにする（連射入力のため）
; ====================================================================================================
GET_CONTROL:
    CALL KILBUF                     ; BIOS キーバッファクリア
    OR A                            ; キャリーフラグクリア

    ; ■プレイヤー操作データ(STICK)を取得
    LD A,0                          ; A <- ジョイスティック番号=0(キーボード)
    CALL GTSTCK                     ; BIOS ジョイスティックの状態取得
                                    ; - Aレジスタに入力値が設定されている
    LD B,A                          ; B <- A

    PUSH BC                         ; BCレジスタを退避
    LD A,1                          ; A <- ジョイスティック番号=1(パッド1)
    CALL GTSTCK                     ; ジョイスティック入力取得
    POP BC                          ; BCレジスタを復帰
    OR B                            ; A=A OR B
                                    ; - キーボードとパッドの入力の OR を取る
                                    ;   最大で15となる

    LD (INPUT_BUFF_STICK), A        ; 現在の入力値を保存


    ; ■プレイヤー操作データ(STRIG)を取得
    LD D,0                          ; D <- 0(入力OFF)
    XOR A                           ; A <- ジョイスティック番号=0(キーボード)
    CALL GTTRIG                     ; BIOS トリガボタンの状態取得
                                    ; - $00 = 押されていない
                                    ; - $0FF = 押されている
    OR A
    JR Z,GET_CONTROL_L1
    LD D,1                          ; D <- 1(入力ON)

GET_CONTROL_L1:
    LD E,0                          ; E <- 0(入力OFF)
    LD A,1                          ; A <- ジョイスティック番号=0(パッド1)
    CALL GTTRIG                     ; BIOS トリガボタンの状態取得
    OR A
    JR Z,GET_CONTROL_L2
    LD E,1                          ; E <- 1(入力ON)

GET_CONTROL_L2:
    LD A,D                          ; A=D OR E(キーボードとジョイスティックのORを取る)
    OR E                            

    OR A
    JR Z,GET_CONTROL_L3             ; ゼロ(未入力)の場合は入力状態とバッファをクリア

    LD A,(INPUT_BUFF_STRIG+1)       ; バッファを取得
    OR A                            
    RET NZ                          ; バッファが入力中なら何もせず終了
    LD A,1                          ; 入力状態とバッファをONにする

GET_CONTROL_L3:
    LD (INPUT_BUFF_STRIG+1),A       ; バッファをクリア
    LD (INPUT_BUFF_STRIG), A        ; 入力状態をクリア
    RET


; ====================================================================================================
; トリガ入力状態を返却する
; 押しっぱなし防止のため、返却時に入力状態をクリアする
; IN  : NONE
; OUT : A(0=未入力、1=入力)
; Zフラグでも判定可能
; ====================================================================================================
GET_STRIG:
    LD HL,INPUT_BUFF_STRIG
    LD A,(HL)
    OR A
    LD (HL),0
    RET


; ====================================================================================================
; キーマトリクス取得処理
; KEYBUFFに押されたキーのキーマトリクスの位置を上位(行)＋下位(列)で格納する（例："A"キー=$36)
; 押しっぱなしにした場合は次フレームからゼロとなる
; ====================================================================================================
GET_KEYMATRIX:

    XOR A
    LD (KEYBUFF),A                  ; キー入力バッファにキーオフを設定
    LD B,6                          ; ループ回数
    LD E,KEY_OFF                    ; キー入力有無判定フラグ

GET_KEYMATRIX_l1:
    LD A,B                          ; A <- キーマトリクスのスキャン対象行
    INC A                           ; ループ回数(1〜6)+1とする                                    
    
    PUSH BC
    CALL GET_KEYMATRIX_SUB          ; 入力キーの情報をバッファに設定
    POP BC
    DJNZ GET_KEYMATRIX_l1

    LD A,KEY_ON
    AND E
    RET NZ

    XOR A
    LD (KEYBUFF_SV),A               ; キー入力バッファSVにOFFを設定

    RET


GET_KEYMATRIX_SUB:
    LD D,A                          ; Dレジスタには入力キーのマトリクス座標を格納する
                                    ; 上位4ビット = 行、最初にスキャン対象行を設定しておく
                                    ; 下位4ビット = 列、ループ処理の中でBレジスタ-1の値を設定
    SLA D                           ; 4ビット左にシフト
    SLA D
    SLA D
    SLA D

    LD B,8                          ; 0〜7ビットをスキャンするためのループ回数

    CALL SNSMAT                     ; BIOS キーマトリクススキャン
    LD C,A                          ; C <- SNSMATのスキャン結果
 
GET_KEYMATRIX_SUB_L1:
    RLC C                           ; ループ処理の中では、Bレジスタの値(8〜1)をキーマトリクスの列として扱いたいため、
                                    ; スキャンデータを左ローテートし、bit0とCフラグにbit7の値を設定する
    JR C,GET_KEYMATRIX_SUB_L3       ; ビットが立っている=キーが押されていないならL2へ

    ; ■キーが押されているときの処理
    ;   キー入力バッファSVの値を読み、OFFの時だけキー入力バッファをONにする
    ;   キー入力バッファSVがONの時は押しっぱなしなので、キー入力バッファをOFFにする
    LD E,KEY_ON                     ; キー入力フラグをON
    LD A,(KEYBUFF_SV)               ; A <- キー入力バッファSV
    OR A
    JR Z,GET_KEYMATRIX_SUB_L12      ; キー入力バッファSVがOFFの時はL12へ

    XOR A
    LD (KEYBUFF),A                  ; キー入力バッファにOFFを設定
                                    ; キー入力バッファSVはそのままで良いので何もしない

    JR GET_KEYMATRIX_SUB_L3

GET_KEYMATRIX_SUB_L12:
    ; ■押されたキーの状態を保存
    LD A,D                          ; A <- キーマトリクスの行(事前に上位4bitに設定済)
    OR B                            ; キーマトリクスの列の値とのORを取り、下位4bitに設定
    DEC A                           ; キーマトリクスの列(=Bレジスタ)は1〜8の範囲なので、-1する

    LD (KEYBUFF),A                 ; キーマトリクスの行列を入力キー情報としてキー入力バッファに格納
                                    ; Dレジスタは事前にキースキャン対象行が設定されている
    LD (KEYBUFF_SV),A
    JR GET_KEYMATRIX_SUB_L3

GET_KEYMATRIX_SUB_L3:
    DJNZ GET_KEYMATRIX_SUB_L1

    RET


; ====================================================================================================
; オフスクリーン初期化
; ====================================================================================================
CLEAR_OFFSCREEN:
    LD B,32*24/4
    LD HL,OFFSCREEN

CLEAR_OFFSCREEN_L1:
    LD (HL),$20
    INC HL
    LD (HL),$20
    INC HL
    LD (HL),$20
    INC HL
    LD (HL),$20
    INC HL
    DJNZ CLEAR_OFFSCREEN_L1

CLEAR_OFFSCREEN_EXIT:
    RET


; ====================================================================================================
; オフスクリーンの内容をVRAMに転送
; ====================================================================================================
DRAW_VRAM:
    LD HL,OFFSCREEN
    LD DE,PTN_NAME_ADDR
    LD BC,32*24
    CALL LDIRVM

DRAW_VRAM_EXIT:
    RET


; ====================================================================================================
; １文字表示サブルーチン
; IN  : HL = オフスクリーンのオフセットアドレス
;       A  = 表示するデータ
; ====================================================================================================
PUTSTR:
    PUSH DE
    PUSH HL
    LD DE,OFFSCREEN                 ; DE <- オフスクリーンバッファの先頭アドレス
    ADD HL,DE                       ; DE=DE+HL
    LD (HL),A                       ; オフスクリーンバッファに設定

PUTSTR_EXIT:
    POP HL
    POP DE
    RET


; ====================================================================================================
; 文字列表示サブルーチン
; IN  : HL = 文字データのアドレス
; ====================================================================================================
PRTSTR:
    PUSH BC
    PUSH DE

    LD C,(HL)                       ; BC <- オフスクリーンオフセット値
    INC HL
    LD B,(HL)

    INC HL                          ; HL <- 文字列データの先頭アドレス
    PUSH HL                         ; HL -> DE
    POP DE                          ; ここでDEに文字列データの先頭アドレスが設定される

    ; ■オフスクリーンの表示開始アドレスを設定
    LD HL,BC

PRTSTR_L1:
	LD A,(DE)				        ; AレジスタにDEレジスタの示すアドレスの文字列データを取得
	OR 0					        ; 0かどうか
    JR Z,PRTSTR_END			        ; 0の場合はPRTENDへ

    CALL PUTSTR

	INC HL					        ; HL=HL+1
    INC DE					        ; DE=DE+1
    JR PRTSTR_L1

PRTSTR_END:
    POP DE
    POP BC
	RET


; ====================================================================================================
; 16進数表示サブルーチン
; IN  : A = 表示対象データ
;       HL = オフスクリーンバッファの表示オフセット値
; ====================================================================================================
PRTHEX:
    PUSH AF
    ; ■オフスクリーンバッファの設定先アドレス算出
    LD DE,OFFSCREEN                 ; DE <- オフスクリーンバッファの先頭アドレス
    ADD HL,DE                       ; HL=HL+DE

    ; ■表示対象データの上位4ビットに対する表示文字コード算出
    PUSH AF                         ; AFレジスタを一旦スタックに退避
    SRL A                           ; 右シフトx4
    SRL A
    SRL A
    SRL A
    CALL PRTHEX_GETCHR              ; Aレジスタの値からキャラクタコードを求める

    LD (HL),A                       ; オフスクリーンバッファに設定

    ; ■VRAMアドレスをインクリメント
    INC HL

    ; ■表示対象データの下位4ビットに対する表示文字コード算出
    POP AF                          ; AFレジスタをスタックから復帰
    AND @00001111                   ; 下位4ビットを取り出し
    CALL PRTHEX_GETCHR              ; Aレジスタの値からキャラクタコードを求める

    LD (HL),A                       ; オフスクリーンバッファに設定

PRTHEX_EXIT:
    POP AF
    RET

PRTHEX_GETCHR:
    OR A                            ; キャリーフラグリセット
    CP 10                           ; A < 10の場合はキャリーフラグが立つ
    JR C,PRTHEX_GETCHR_L1
    ADD A,$37                       ; A〜F
    RET

PRTHEX_GETCHR_L1:
    ADD A,$30                       ; 0〜9
    RET


; ====================================================================================================
; 絶対値減算サブルーチン
; IN  : A = 値１
;       B = 値２
; OUT : A = A-Bの絶対値
; ====================================================================================================
ABS_SUB:
    OR A
    SUB B                           ; A=A-B
    JP M,ABS_SUB_L1                 ; マイナスだったらABS_SUB_L1へ
    RET

ABS_SUB_L1:
    NEG                             ; 上記の結果を正負反転
	ADD A,$FF                       ; A=A+$FF
    ADD A,1                         ; さらに1加算して一巡させる
    RET


; ====================================================================================================
; アドレステーブルからのデータ取得サブルーチン
; IN  : HL = アドレステーブルのアドレス
;       A = 要素数(0～)
; OUT : DE = アドレステーブルから取得したデータ
; ====================================================================================================
GET_ADDR_TBL:
    RLCA                            ; A=A*2
    LD D,0                          ; DE <- アドレスのオフセット値
    LD E,A

    ADD HL,DE                       ; HL=HL+DE

    LD E,(HL)                       ; DE <- テーブルの値
    INC HL
    LD D,(HL)

GET_ADDR_TBL_EXIT:
    RET


; ====================================================================================================
; アドレステーブルによるジャンプ処理
; ジャンプ先の処理でRETすると、ここの処理に戻ってきます。
; IN  : HL = 対象テーブルの先頭アドレス
;       A = インデックスNo(0～)
; ====================================================================================================
TBL_JP:
    CALL GET_ADDR_TBL               ; アドレステーブルからデータ取得

    LD HL,TBL_JP_EXIT               ; 戻り先のアドレスをスタックに設定
    PUSH HL

    EX DE,HL                        ; HL <-> DE

    JP (HL)

TBL_JP_EXIT:
    RET


; ====================================================================================================
; 乱数初期化サブルーチン
; ====================================================================================================
INIT_RND:
    LD A,(INTCNT)
    LD (RND_WK),A                   ; 乱数のシード値を設定

INIT_RND_EXIT:
    RET


; ====================================================================================================
; 乱数取得サブルーチン
; 事前にINIT_RNDを実行しておくこと
; OUT : A = 0〜255の範囲の乱数
; ====================================================================================================
GET_RND:
    PUSH BC
    
    LD A,(RND_WK)                   ; 乱数のシード値を乱数ワークエリアから取得
    LD B,A
    LD A,B

    ADD A,A                         ; A=A*5
    ADD A,A                         ;
    ADD A,B                         ;

    ADD A,123                       ; 123を加える
    LD (RND_WK),A                   ; 乱数ワークエリアに保存

GET_RND_EXIT:
    POP BC
    RET


; ====================================================================================================
; 画面クリアサブルーチン
; HL,BC,Aレジスタを破壊します
; ====================================================================================================
SCREEN_CLRAR:
    LD HL,$1800+32*2                ; 書き込み開始アドレス
    LD BC,32*22                     ; 書き込みデータ長
    LD A,$20                        ; 書き込むデータ
    CALL FILVRM                     ; BIOS VRAM指定領域同一データ転送

SCREEN_CLRAR_EXIT:
    RET


; ------------------------------------------------------------------------------------------------
; 二進化十進値加算サブルーチン（6桁）
; IN  : HL = 加算対象データの先頭アドレス
;       DE = 加算値(BCD形式で4桁、1/100の値とする。例：12000＝0120)
; ------------------------------------------------------------------------------------------------
ADD_BCD_6:
    INC HL                      ; 6桁の1〜2桁目を最初に処理するため、アドレスを進めておく
    CALL ADD_BCD_4              ; 1〜4桁目までは既存の処理で加算

    ; ■5〜6桁目の加算
    ;   加算値は4桁までしかないので、桁繰り上がりの加算だけ行う
    RET NC                      ; キャリーが立っていなければ終了
    DEC HL
    LD A,(HL)                   ; 5〜6桁目に1加算
    INC A
    DAA
    LD (HL),A
    RET

; ------------------------------------------------------------------------------------------------
; 二進化十進値加算サブルーチン（4桁）
; IN  : HL = 加算対象データの先頭アドレス
;       DE = 加算値(BCD形式で4桁、1/100の値とする。例：12000＝0120)
; ------------------------------------------------------------------------------------------------
ADD_BCD_4:
    ; ■1〜2桁目の加算
    INC HL
	LD A,E					    ; AレジスタにEレジスタの値をロード
    ADD A,(HL)			        ; Aレジスタの値に(HL)の値を加算
    						    ; 桁溢れした場合はキャリーフラグが立つ
    DAA						    ; Aレジスタの値を内部10進に補正
    LD (HL),A				    ; Aレジスタの値を(HL)に格納
    
    ; ■3〜4桁目の加算
    DEC HL
    LD A,D					    ; AレジスタにDレジスタの値をロード
    ADC A,(HL)			        ; Aレジスタの値に(HL)＋キャリーフラグを加算
    						    ; 桁溢れした場合はキャリーフラグが立つが無視する
    DAA						    ; Aレジスタの値を内部10進に補正
    LD (HL),A				    ; Aレジスタの値を(HL)に格納

    RET


; ------------------------------------------------------------------------------------------------
; 内部10進数表示サブルーチン
; オフスクリーンバッファの指定したオフセットアドレスにBCD形式のデータを数値として表示する
; 表示する値のデータは、以下のように格納されたものとする
; [上位桁][下位桁] [上位桁][下位桁]…
; IN  : B = 表示するデータのバイト数(1バイト=2桁)
;       C = 表示桁を埋める文字コード($00=スペースと扱う)
;       DE = 表示するデータのアドレス
;       HL = オフスクリーンバッファの設定先オフセットアドレス
; ------------------------------------------------------------------------------------------------
PRTBCD:
    LD A,C                      ; A <- 表示桁を埋める文字コード
    OR A
    JR NZ,PRTBCD_L1             ; 表示桁を埋める文字コードが$00でなければL1へ
    LD A,$20                    ; $00ならスペースにする

PRTBCD_L1:
    PUSH BC
    PUSH DE
    PUSH HL

    LD A,B
    ADD A,A                     ; 表示文字数を算出 (表示桁*2)
    LD B,A                      ; BC(繰り返し数) = 表示文字数
    LD DE,OFFSCREEN
    ADD HL,DE
PRTBCD_L2:
    LD (HL),C                   ; 表示桁分、文字を埋める
    INC HL
    DJNZ PRTBCD_L2
    DEC HL
    LD (HL),$30                 ; 末尾はゼロ固定表示

    POP HL
    POP DE
    POP BC

    LD C,0                      ; ゼロ表示フラグ初期化
PRTBCD_L3:

    LD A,(DE)
    PUSH DE
    PUSH HL
	CALL PUTBCD 			    ; データを表示
    POP HL
    POP DE

	INC DE					    ; DEレジスタの値を1加算(＝データの次のアドレスが設定される)
    INC HL
    INC HL					    ; HLレジスタの値を2加算(＝表示位置を2つ右に移動)

    DJNZ PRTBCD_L3              ; B=B-1、ゼロでなければ繰り返す

	RET

; ------------------------------------------------------------------------------------------------
; 内部10進数表示サブルーチン(1バイト)
; DE,HLレジスタを破壊します
; 先頭ゼロ表示を表示しないとした場合、先頭ゼロは表示をスキップします(カウントダウン時は予め表示エリアの初期化が必要なので注意)
; IN  : A  = 表示するデータ(BCD値)
;       C  = 先頭ゼロ表示(0=表示しない、1=表示する)
;       HL = 表示先のオフスクリーンオフセットアドレス
; ------------------------------------------------------------------------------------------------
PUTBCD:
    LD DE,OFFSCREEN
    ADD HL,DE                   ; HLレジスタにオフセットアドレス＋オフスクリーンバッファ先頭アドレスを設定

	; ■上1桁の処理
    PUSH AF
    SRL A					    ; Aレジスタの値を4回右シフトして、上位4ビットを取り出す
    SRL A
    SRL A
    SRL A
    CALL PUTBCD_L1			    ; オフスクリーンバッファにデータ設定
    
	; ■下1桁の処理
    POP AF
    INC HL					    ; HLレジスタの値を1加算(＝データ表示位置を1つ右に移動)

PUTBCD_L1:
	; ■オフスクリーンバッファにデータ設定
	AND $0F				        ; 上位4ビットをゼロにする(=下位4ビットの値だけ取り出す)
    OR A
    JR NZ,PUTBCD_L2             ; 値がゼロ以外の場合はL2へ

    INC C                       ; 値がゼロの時はゼロ表示フラグを判定
    DEC C
    RET Z                       ; ゼロ表示フラグがOFFの時は抜ける

PUTBCD_L2:
    INC C                       ; 以降のゼロは表示させるので、ゼロ表示フラグを+1
    ADD A,$30				    ; 値にキャラクタコード&H30('0')を加える
    LD (HL),A                   ; オフスクリーンバッファにデータを設定

PUTBCD_EXIT:
    RET

; ====================================================================================================
; HLレジスタにAレジスタを加算するサブルーチン
; IN  : A = 加算する値
;     : HL = 加算される値
; OUT : HL = 計算後の値
; ====================================================================================================
ADD_HL_A:
    ADD A,L
    JR NC,ADD_HL_A_L1
    INC H
ADD_HL_A_L1:
    LD L,A
    RET


; ====================================================================================================
; ワークエリア
; プログラム起動時にcrtでゼロでramに設定される 
; ====================================================================================================
SECTION bss_user

; ■乱数ワークエリア
RND_WK:
    DB 0

; ■入力バッファ(STICK)
; +0 : 現在の入力値
; +1 : 前回の入力値
INPUT_BUFF_STICK:
    DEFS 2

; ■入力バッファ(STRIG)
; +0 : 現在の入力値
; +1 : 前回の入力値
INPUT_BUFF_STRIG:
    DEFS 2

; ■キー入力バッファ
KEYBUFF:
    DB  $00
KEYBUFF_SV:
    DB  $00

; ■オフスクリーンバッファ
OFFSCREEN:
    DEFS 32*24
